// File_Riff - Info for RIFF files
// Copyright (C) 2002-2006 Jerome Martinez, Zen@MediaArea.net
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//
// Information about RIFF files
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//
// 2005-11-07, Zen@MediaArea.net
// Adapted to new File__Base
//
// 2005-10-16, Zen@MediaArea.net
// Better detection and handling of DivX container
// RIFF files with not-padded (INFO) sub chunks can't be parsed
//
// 2005-05-18, Zen@MediaArea.net
// Should work with files having broken video stream header
//
// 2005-03-12, Zen@MediaArea.net
// Complete rewriting : without win32 API
//
// 2004-10-08, Zen@MediaArea.net
// Release the file after analized it
//
// 2003-05-31, Zen@MediaArea.net
// Adapt to new architecture and being LGPL
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//---------------------------------------------------------------------------
// Compilation condition
#if defined(MEDIAINFO_RIFF_YES) || (!(defined(MEDIAINFO_VIDEO_NO) && defined(MEDIAINFO_AUDIO_NO)) && !defined(MEDIAINFO_RIFF_NO))
//---------------------------------------------------------------------------

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// RIFF format
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//
//
// RIFF (size) AVI       <- not more than 1 GB in size
//     LIST (size) hdrl
//         avih (size) (datas)
//         LIST (size) strl
//             strh (size) (datas)
//             strf (size) (datas)
//             indx (size) (datas)   <- size may vary, should be sector sized
//         LIST (size) odml
//             dmlh (size) (datas)
//         JUNK (size) (datas)      <- fill to align to sector - 12
//     LIST (size) movi  <- aligned on sector - 12
//         00dc (size)       <- sector aligned
//         01wb (size)       <- sector aligned
//         ix00 (size)       <- sector aligned
//     idx1 (size)       <- sector aligned
// RIFF (size) AVIX
//     JUNK (size)           <- fill to align to sector -12
//     LIST (size) movi
//         00dc (size)       <- sector aligned
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//---------------------------------------------------------------------------
#include <wx/wxprec.h>
#ifdef __BORLANDC__
    #pragma hdrstop
#endif
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
#include <ZenLib/Utils.h>
#include <ZenLib/Ztring.h>
#include "MediaInfo/Multiple/File_Riff.h"
using namespace ZenLib;
//---------------------------------------------------------------------------

namespace MediaInfoLib
{

//Microsoft structures
typedef struct {
    int32u  MicroSecPerFrame;     // frame display rate (or 0L)
    int32u  MaxBytesPerSec;       // max. transfer rate
    int32u  PaddingGranularity;   // pad to multiples of this size; normally 2K.
    int32u  Flags;                // the ever-present flags
    #define AVIF_HASINDEX        0x00000010 // Index at end of file?
    #define AVIF_MUSTUSEINDEX    0x00000020
    #define AVIF_ISINTERLEAVED   0x00000100
    #define AVIF_TRUSTCKTYPE     0x00000800 // Use CKType to find key frames
    #define AVIF_WASCAPTUREFILE  0x00010000
    #define AVIF_COPYRIGHTED     0x00020000
    int32u  TotalFrames;          // # frames in first movi list
    int32u  InitialFrames;
    int32u  Streams;
    int32u  SuggestedBufferSize;
    int32u  Width;
    int32u  Height;
    int32u  Reserved[4];
    } AviMainHeader; // "avih" header

typedef struct {
   int32u  GrandFrames;          // total number of frames in the file
   } DmlHeader; // "dmlh" header

typedef struct {
    int32u fccType;
    int32u fccHandler;
    int32u Flags;
    int32u Caps;
    int16u Priority;
    int16u Language;
    int32u Scale;
    int32u Rate; // Rate/Scale is stream tick rate in ticks/sec
    int32u Start;
    int32u Length;
    int32u InitialFrames;
    int32u SuggestedBufferSize;
    int32u Quality;
    int32u SampleSize;
    struct {
        int16s Left;
        int16s Top;
        int16s Right;
        int16s Bottom;
        } Frame;
    int32u EditCount;
    int32u FormatChangeCount;
    char  Name[64];
} StreamInfo; //strh header

typedef struct {
   int32s  Size;
   int32s  Width;
   int32s  Height;
   int16u  Planes;
   int16u  BitCount;
   int32u  Compression;
   int32u  SizeImage;
   int32s  XPelsPerMeter;
   int32s  YPelsPerMeter;
   int32u  ClrUsed;
   int32u  ClrImportant;
} VideoInfo;

typedef struct {
    int16u FormatTag;
    int16u Channels;
    int32u SamplesPerSec;
    int32u AvgBytesPerSec;
    int16u BlockAlign;
    int16u BitsPerSample;
    int16u Size;
} AudioInfo;


//---------------------------------------------------------------------------
//To clarify the code
#define FIND_BEGIN() \
while (!ShouldStop && Offset<=Element_Offset[Level]+8+Element_Size[Level]) \
{ \
    Level++; \
    Element_Offset[Level]=Offset; \
    Element_Size  [Level]=LittleEndian2int32u((const char*)Begin+Offset+4); \
    Element_List  [Level]=CC4(Begin+Offset+8); \
    if (0) \
    {

//-For each Atom
#define FIND_ATOM_BEGIN(_ATOM) \
    } \
    else if (CC4(Begin+Offset)==CC4(_ATOM)) \
    { \
        Offset+=8;

//-For each LIST
#define FIND_LIST_BEGIN(_LIST) \
    } \
    else if ((CC4(Begin+Offset)==CC4("LIST") || CC4(Begin+Offset)==CC4("RIFF")) && CC4(Begin+Offset+8)==CC4(_LIST)) \
    { \
        Offset+=12; \
        while (!ShouldStop && Offset<=Element_Offset[Level]+8+Element_Size[Level]) \
        { \
            Level++; \
            Element_Offset[Level]=Offset; \
            Element_Size  [Level]=LittleEndian2int32u((const char*)Begin+Offset+4); \
            Element_List  [Level]=CC4(Begin+Offset+8); \
            if (0) \
            {

//-
#define FIND_END() \
    } \
    Offset=Element_Offset[Level]+8+Element_Size[Level]; \
    Level--; \
    if (Offset>=Begin_Size) \
        ShouldStop=true; \
}

//---------------------------------------------------------------------------
int File_Riff::Read(const int8u* Begin, size_t Begin_Size, const int8u* End, size_t End_Size, int64u FileSize)
{
    //Integrity test
    if (Begin_Size<512)
        return -2;
    if (CC4(Begin)!=CC4("RIFF"))
        return -1;

    size_t Offset=0;
	bool ShouldStop=false;
    size_t Level=0;
    size_t Element_Offset[10]; for (size_t Pos=0; Pos<10; Pos++) Element_Offset[Pos]=0;
    size_t Element_Size[10];   for (size_t Pos=0; Pos<10; Pos++) Element_Size  [Pos]=0;
    int32u Element_List[10];   for (size_t Pos=0; Pos<10; Pos++) Element_List  [Pos]=0; //4CC of the list type





    FIND_BEGIN()
        FIND_LIST_BEGIN("AVI ")
                FIND_LIST_BEGIN("hdrl")
                        FIND_ATOM_BEGIN("avih")
                            int A=0;
                        FIND_LIST_BEGIN("strl")
                                FIND_ATOM_BEGIN("strh")
                                    int A=0;
                                FIND_ATOM_BEGIN("strf")
                                    int A=0;
                            FIND_END()
                    FIND_END()
            FIND_END()
    FIND_END()

/*
    FIND_BEGIN()
        FIND_LIST_BEGIN("AVI ")
            FIND_BEGIN()
                FIND_LIST_BEGIN("hdrl")
                    FIND_BEGIN()
                        FIND_LIST_BEGIN("strl")
                            int A=0;
                        FIND_LIST_END()
                    FIND_END()
                FIND_LIST_END()
            FIND_END()
        FIND_LIST_END()
    FIND_END()
*/

/*
    while (!ShouldStop)
    {
        if (0)
        {
        }
        else if ((CC4(Begin+Offset)==CC4("LIST") || CC4(Begin+Offset)==CC4("RIFF")= && Element_List[Level]==CC4("AVI "))
        {
            Level++;
            Element_Offset[Level]=Offset;
            Element_Size  [Level]=LittleEndian2int32u((const char*)Begin+Offset+4);
            Element_List  [Level]=CC4(Begin+Offset+8);
            Offset+=12;
            int A=0;
            Offset=Element_Offset[Level]+Element_Size[Level];
            Level--;
            if (Offset>=Begin_Size)
                ShouldStop=true;
        }
    }
*/
    return 1;

/*
            else if (Element_List[Level]==CC4("LIST") || CC4(Begin+Offset)==CC4("RIFF")) //LIST hdrl
            {
                //This is a list
                Level++;
                Element_Offset[Level]=Offset;
                Element_Size  [Level]=LittleEndian2int32u((const char*)Begin+Offset+4);
                Element_List  [Level]=CC4(Begin+Offset+8);
                Offset+=12;
                if (0)
                {
                }
                else if (Element_List[Level]==CC4("hdrl")) //hdrl
                {
                    Level++;
                    Element_Offset[Level]=Offset;
                    Element_Size  [Level]=LittleEndian2int32u((const char*)Begin+Offset+4);
                    Element_List  [Level]=CC4(Begin+Offset+8);
                    Offset+=12;
                    while (!ShouldStop && Element_Offset[Level]+Element_Size[Level]<=Element_Offset[Level-1])
                    {
                        Element_Offset[Level]=Offset;
                        Element_Size  [Level]=LittleEndian2int32u((const char*)Begin+Offset+4);
                        Element_List  [Level]=0;
                        Offset+=8;
                        if (0)
                        {
                        }
                        else if (CC4(Begin+Offset)==CC4("avih"))
                        {
                            //Todo
                        }
                        Offset+=Element_Size[Level];
                    }
                    Offset=Element_Offset[Level]+Element_Size[Level];
                    Level--;
                }
                Offset=Element_Offset[Level]+Element_Size[Level];
            }
            Offset=Element_Offset[Level]+Element_Size[Level];
            Level--;
            return 1;
        }
        else
            return -1;
    }
*/
/*
    //Wave format
         if (CC4(Begin+Offset)==CC4("WAVE"))
    {
        Stream_Prepare(Stream_General);
        General[0](_T("Format"))=_T("WAV");
        General[0](_T("Format/String"))=_T("Wave");
        General[0](_T("Format/Extensions"))=_T("WAV");

        size_t Wave_Offset=Offset+4; //kind
        if (CC4(Begin+Wave_Offset)==CC4("fmt_") || CC4(Begin+Wave_Offset)==CC4("fmt "))
        {
            size_t Audio_Count=Stream_Prepare(Stream_Audio);
            AudioInfo* StreamFormat=(AudioInfo*)(Begin+Wave_Offset+8);
            Audio[0](_T("Codec")).From_Number(StreamFormat->FormatTag, 16);
            Audio[0](_T("Channel(s)")).From_Number(StreamFormat->Channels);
            Audio[0](_T("SamplingRate")).From_Number(StreamFormat->SamplesPerSec);
            Audio[0](_T("BitRate")).From_Number(StreamFormat->AvgBytesPerSec*8);
            Audio_Count++;
            //WAV is CBR --> We can calculate PlayTime
            Audio[0](_T("BitRate_Mode"))=_T("CBR");
            if (Audio[0](_T("BitRate")).To_int64u())
                General[0](_T("PlayTime")).From_Number(General[0](_T("FileSize")).To_int64u()*1000/Audio[0](_T("BitRate")).To_int64u()*8);
        }
        return 1;
    }

    //AVI format
    else if (CC4(Begin+Offset)==CC4("AVI "))
    {
        Stream_Prepare(Stream_General);
        General[0](_T("Format"))=_T("AVI");
        General[0](_T("Format/String"))=_T("Audio Video Interleave");
        General[0](_T("Format/Extensions"))=_T("AVI");

        size_t Avi_Offset=Offset+4; //Kind
        int32u TotalFrame=0; //Count of frame in all AVI file (with odml too)
        while (Avi_Offset+512<Begin_Size)
        {
            size_t Avi_XXX_Size=LittleEndian2int32u((const char*)Begin+Avi_Offset+4);
            if (CC4(Begin+Avi_Offset)==CC4("LIST"))
            {
                size_t List_Offset=Avi_Offset+8; //List kind
                if (CC4(Begin+List_Offset)==CC4("hdrl"))
                {
                    size_t Hdrl_Offset=List_Offset+4; //hdlr Kind
                    while (Hdrl_Offset<Avi_Offset+Avi_XXX_Size)
                    {
                        size_t Hdrl_XXX_Size=LittleEndian2int32u((const char*)Begin+Hdrl_Offset+4);
                             if (CC4(Begin+Hdrl_Offset)==CC4("avih"))
                        {
                            AviMainHeader* AviMainInfo=(AviMainHeader*)(Begin+Hdrl_Offset+8);
                            TotalFrame=AviMainInfo->TotalFrames;
                        }
                        else if (CC4(Begin+Hdrl_Offset)==CC4("LIST"))
                        {
                            size_t Hdrl_List_Offset=Hdrl_Offset+8;
                                 if (CC4(Begin+Hdrl_List_Offset)==CC4("strl"))
                            {
                                size_t Hdrl_Strl_Offset=Hdrl_List_Offset+4; //strl Kind
                                StreamInfo* AviInfo=NULL;
                                while (Hdrl_Strl_Offset<Hdrl_Offset+Hdrl_XXX_Size)
                                {
                                    size_t Hdrl_Strl_XXX_Size=LittleEndian2int32u((const char*)Begin+Hdrl_Strl_Offset+4);
                                         if (CC4(Begin+Hdrl_Strl_Offset)==CC4("strh"))
                                    {
                                        AviInfo=(StreamInfo*)(Begin+Hdrl_Strl_Offset+8);
                                    }
                                    else if (CC4(Begin+Hdrl_Strl_Offset)==CC4("strf"))
                                    {
                                        //Stream Format
                                        //-Video : classic or DV
                                             if (AviInfo->fccType==LittleEndian2int32u("vids"))
                                        {
                                            VideoInfo* StreamFormat=(VideoInfo*)(Begin+Hdrl_Strl_Offset+8);
                                            size_t Video_Count=Stream_Prepare(Stream_Video);
                                            if ((StreamFormat->Compression&0x000000FF)!=0x000000FF
                                             && (StreamFormat->Compression&0x0000FF00)!=0x0000FF00
                                             && (StreamFormat->Compression&0x00FF0000)!=0x00FF0000
                                             && (StreamFormat->Compression&0xFF000000)!=0xFF000000)
                                            {
                                                //Good version of AVI files
                                                Video[Video_Count](_T("Codec")).From_Local((char*)&(StreamFormat->Compression), 0, 4);
                                                Video[Video_Count](_T("Width")).From_Number(StreamFormat->Width);
                                                Video[Video_Count](_T("Height")).From_Number(StreamFormat->Height);
                                            }
                                            else if (AviInfo)
                                            {
                                                //Some Stream headers are broken, must use AVISTREAMINFOA structure instead of AVIFILEINFOA
                                                Video[Video_Count](_T("Codec")).From_Local((char*)&(AviInfo->fccHandler), 0, 4);
                                                Video[Video_Count](_T("Width")).From_Number(AviInfo->Frame.Right);
                                                Video[Video_Count](_T("Height")).From_Number(AviInfo->Frame.Bottom);
                                            }
                                            if (AviInfo && AviInfo->Scale)
                                                Video[Video_Count](_T("FrameRate")).From_Number(((float)AviInfo->Rate)/(float)AviInfo->Scale, 3);
                                        }
                                        //-DV Type 1
                                        else if (AviInfo->fccType==LittleEndian2int32u("iavs"))
                                        {
                                            size_t Video_Count=Stream_Prepare(Stream_Video);
                                            if (AviInfo && AviInfo->Scale)
                                            {
                                                Video[Video_Count](_T("FrameRate")).From_Number(((float)AviInfo->Rate)/(float)AviInfo->Scale, 3);
                                                Video[Video_Count](_T("Codec")).From_Local((char*)&(AviInfo->fccHandler), 0, 4);
                                                     if (Video[Video_Count](_T("Codec"))==_T("dvsd"))
                                                {
                                                    Video[Video_Count](_T("Width"))=_T("720");
                                                        if(Video[Video_Count](_T("FrameRate"))==_T("25.000"))
                                                        Video[Video_Count](_T("Height"))=_T("576");
                                                    else if (Video[Video_Count](_T("FrameRate"))==_T("29.970"))
                                                        Video[Video_Count](_T("Height"))=_T("480");
                                                }
                                                else if (Video[Video_Count](_T("Codec"))==_T("dvhd"))
                                                {
                                                    Video[Video_Count](_T("Width"))=_T("1440");
                                                        if(Video[Video_Count](_T("FrameRate"))==_T("25.000"))
                                                        Video[Video_Count](_T("Height"))=_T("1152");
                                                    else if (Video[Video_Count](_T("FrameRate"))==_T("30.000"))
                                                        Video[Video_Count](_T("Height"))=_T("960");
                                                }
                                            }
                                        }
                                        //-Audio
                                        else if (AviInfo->fccType==LittleEndian2int32u("auds"))
                                        {
                                            AudioInfo* StreamFormat=(AudioInfo*)(Begin+Hdrl_Strl_Offset+8);
                                            size_t Audio_Count=Stream_Prepare(Stream_Audio);
                                            Audio[Audio_Count](_T("Codec")).From_Number(StreamFormat->FormatTag, 16);
                                            Audio[Audio_Count](_T("Channel(s)")).From_Number(StreamFormat->Channels);
                                            if (Audio[Audio_Count](_T("Channel(s)"))==_T("5"))
                                                Audio[Audio_Count](_T("Channel(s)"))=_T("6"); //Some workaround about AC3 (for example) with subwoofer
                                            Audio[Audio_Count](_T("SamplingRate")).From_Number(StreamFormat->SamplesPerSec);
                                            Audio[Audio_Count](_T("BitRate")).From_Number(StreamFormat->AvgBytesPerSec*8);
                                        }
                                        //-Audio, Midi
                                        else if (AviInfo->fccType==LittleEndian2int32u("mids"))
                                        {
                                            size_t Audio_Count=Stream_Prepare(Stream_Audio);
                                            Audio[Audio_Count](_T("Codec"))=_T("Midi");
                                        }
                                        //-Text
                                        else if (AviInfo->fccType==LittleEndian2int32u("txts"))
                                        {
                                            size_t Text_Count=Stream_Prepare(Stream_Text);
                                            Text[Text_Count](_T("Codec"))=_T("SRT");
                                        }
                                    }
                                    else if (CC4(Begin+Hdrl_Strl_Offset)==CC4("indx"))
                                    {
                                    }
                                    else if (CC4(Begin+Hdrl_Strl_Offset)==CC4("strn"))
                                    {
                                        //Divx.com hack
                                        Ztring HackKind; HackKind.From_Local((const char*)Begin+Hdrl_Strl_Offset+8, 0, Hdrl_Strl_XXX_Size);
                                        if (HackKind==_T("Subtitle") || HackKind==_T("sub"))
                                        {
                                            if (!Video.empty())
                                                Video.resize(Video.size()-1);
                                            size_t Text_Count=Stream_Prepare(Stream_Text);
                                            Text[Text_Count](_T("Codec"))=_T("DXSB");
                                            General[0](_T("Format"))=_T("DivX");
                                            General[0](_T("Format/String"))=_T("DivX");
                                            General[0](_T("Format/Url"))=_T("http://www.divx.com");
                                            General[0](_T("Format/Extensions"))=_T("DIVX");
                                        }
                                    }
                                    Hdrl_Strl_Offset+=Hdrl_Strl_XXX_Size+(Hdrl_Strl_XXX_Size&1)+8; //To next trunk
                                }
                            }
                            else if (CC4(Begin+Hdrl_List_Offset)==CC4("odml"))
                            {
                                size_t Hdrl_Odml_Offset=Hdrl_List_Offset+4; //strl Kind
                                while (Hdrl_Odml_Offset<Hdrl_Offset+Hdrl_XXX_Size)
                                {
                                    size_t Hdrl_Odml_XXX_Size=LittleEndian2int32u((const char*)Begin+Hdrl_Odml_Offset+4);
                                         if (CC4(Begin+Hdrl_Odml_Offset)==CC4("dmlh"))
                                    {
                                        DmlHeader* Dml=(DmlHeader*)(Begin+Hdrl_Odml_Offset+8);
                                        TotalFrame=Dml->GrandFrames;
                                    }
                                    Hdrl_Odml_Offset+=Hdrl_Odml_XXX_Size+(Hdrl_Odml_XXX_Size&1)+8; //To next trunk
                                }
                            }
                        }
                        else if (CC4(Begin+Hdrl_Offset)==CC4("INAM"))
                            General[0](_T("Title")).From_Local((const char*)Begin+Hdrl_Offset+8, 0, Hdrl_XXX_Size);
                        else if (CC4(Begin+Hdrl_Offset)==CC4("ISFT"))
                            General[0](_T("Encoded_Application")).From_Local((const char*)Begin+Hdrl_Offset+8, 0, Hdrl_XXX_Size);
                        else if (CC4(Begin+Hdrl_Offset)==CC4("ITCH"))
                            General[0](_T("EncodedBy")).From_Local((const char*)Begin+Hdrl_Offset+8, 0, Hdrl_XXX_Size);
                        else if (CC4(Begin+Hdrl_Offset)==CC4("ICMS"))
                            General[0](_T("CommissionedBy")).From_Local((const char*)Begin+Hdrl_Offset+8, 0, Hdrl_XXX_Size);
                        else if (CC4(Begin+Hdrl_Offset)==CC4("ICMS"))
                            General[0](_T("CommissionedBy")).From_Local((const char*)Begin+Hdrl_Offset+8, 0, Hdrl_XXX_Size);
                        else if (CC4(Begin+Hdrl_Offset)==CC4("ISRC"))
                            General[0](_T("Encoded_Original/DistributedBy")).From_Local((const char*)Begin+Hdrl_Offset+8, 0, Hdrl_XXX_Size);
                        else if (CC4(Begin+Hdrl_Offset)==CC4("ISBJ"))
                            General[0](_T("Subject")).From_Local((const char*)Begin+Hdrl_Offset+8, 0, Hdrl_XXX_Size);
                        else if (CC4(Begin+Hdrl_Offset)==CC4("IKEY"))
                            General[0](_T("Keywords")).From_Local((const char*)Begin+Hdrl_Offset+8, 0, Hdrl_XXX_Size);
                        else if (CC4(Begin+Hdrl_Offset)==CC4("IMED"))
                            General[0](_T("Medium")).From_Local((const char*)Begin+Hdrl_Offset+8, 0, Hdrl_XXX_Size);
                        else if (CC4(Begin+Hdrl_Offset)==CC4("IPRD"))
                            General[0](_T("Product")).From_Local((const char*)Begin+Hdrl_Offset+8, 0, Hdrl_XXX_Size);
                        else if (CC4(Begin+Hdrl_Offset)==CC4("ICRD"))
                        {
                            Ztring S;
                            S.From_Local((const char*)Begin+Hdrl_Offset+8, Hdrl_XXX_Size);
                            if (S.size()==10)
                                //The specification says to format the date as: "2005-10-14",
                                //however there should be some tolerance allowed.
                                //Note : no conversion from local to UTC, local time of read is not local time of record.
                                General[0](_T("Written_Date"))=S.substr(0, 4)+S.substr(5, 2)+S.substr(7, 2);
                            else
                                General[0](_T("Written_Date"))=S;
                        }
                        else if (CC4(Begin+Hdrl_Offset)==CC4("IDIT"))
                        {
                            Ztring S;
                            S.From_Local((const char*)Begin+Hdrl_Offset+8, Hdrl_XXX_Size);
                            if (S.size()==10)
                                //The specification says to format the date as: "2005-10-14",
                                //however there should be some tolerance allowed.
                                //Note : no conversion from local to UTC, local time of read is not local time of record.
                                General[0](_T("Mastered_Date"))=S.substr(0, 4)+S.substr(5, 2)+S.substr(7, 2);
                            else
                                General[0](_T("Mastered_Date"))=S;
                        }
                        else if (CC4(Begin+Hdrl_Offset)==CC4("IARL"))
                            General[0](_T("Archival_Location")).From_Local((const char*)Begin+Hdrl_Offset+8, 0, Hdrl_XXX_Size);
                        else if (CC4(Begin+Hdrl_Offset)==CC4("IGNR"))
                            General[0](_T("Genre")).From_Local((const char*)Begin+Hdrl_Offset+8, 0, Hdrl_XXX_Size);
                        else if (CC4(Begin+Hdrl_Offset)==CC4("ICMT"))
                            General[0](_T("Comment")).From_Local((const char*)Begin+Hdrl_Offset+8, 0, Hdrl_XXX_Size);
                        else if (CC4(Begin+Hdrl_Offset)==CC4("ISFT"))
                            General[0](_T("Encoded_Application")).From_Local((const char*)Begin+Hdrl_Offset+8, 0, Hdrl_XXX_Size);
                        else if (CC4(Begin+Hdrl_Offset)==CC4("ISRF"))
                            General[0](_T("Encoded_Original")).From_Local((const char*)Begin+Hdrl_Offset+8, 0, Hdrl_XXX_Size);
                        else if (CC4(Begin+Hdrl_Offset)==CC4("ICOP"))
                            General[0](_T("Copyright")).From_Local((const char*)Begin+Hdrl_Offset+8, 0, Hdrl_XXX_Size);
                        Hdrl_Offset+=Hdrl_XXX_Size+(Hdrl_XXX_Size&1)+8;
                    }
                }
                //Info
                else if (CC4(Begin+List_Offset)==CC4("INFO"))
                {
                    size_t Info_Offset=List_Offset+4; //hdlr Kind
                    while (Info_Offset<Avi_Offset+Avi_XXX_Size)
                    {
                        size_t Info_XXX_Size=LittleEndian2int32u((const char*)Begin+Info_Offset+4);
                             if (CC4(Begin+Info_Offset)==CC4("INAM"))
                            General[0](_T("Title")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("IPRT"))
                            General[0](_T("Track/Position")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("IFRM"))
                            General[0](_T("Album/Track_Total")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("IMUS"))
                            General[0](_T("Composer")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("IWRI"))
                            General[0](_T("WrittenBy")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("IART"))
                            General[0](_T("Director")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("ICNM"))
                            General[0](_T("DirectorOfPhotography")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("IEDT"))
                            General[0](_T("EditedBy")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("IPRO"))
                            General[0](_T("Producer")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("IPDS"))
                            General[0](_T("ProductionDesigner")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("ICDS"))
                            General[0](_T("CostumeDesigner")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("IENG"))
                            General[0](_T("MasteredBy")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("ISTD"))
                            General[0](_T("ProductionStudio")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("IDST"))
                            General[0](_T("DistributedBy")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("ITCH"))
                            General[0](_T("EncodedBy")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("IRIP"))
                            General[0](_T("EncodedBy")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("IENC"))
                            General[0](_T("EncodedBy")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("ICMS"))
                            General[0](_T("CommissionedBy")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("ISRC"))
                            General[0](_T("Encoded_Original/DistributedBy")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("ISBJ"))
                            General[0](_T("Subject")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("IKEY"))
                            General[0](_T("Keywords")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("IRTD"))
                            General[0](_T("LawRating")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("ILNG"))
                            General[0](_T("Language")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("IMED"))
                            General[0](_T("Medium")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("IPRD"))
                            General[0](_T("Product")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("ICNT"))
                            General[0](_T("Country")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("ICRD"))
                        {
                            Ztring S;
                            S.From_Local((const char*)Begin+Info_Offset+8, Info_XXX_Size);
                            if (S.size()==10)
                                //The specification says to format the date as: "2005-10-14",
                                //however there should be some tolerance allowed.
                                //Note : no conversion from local to UTC, local time of read is not local time of record.
                                General[0](_T("Written_Date"))=S.substr(0, 4)+S.substr(5, 2)+S.substr(7, 2);
                            else
                                General[0](_T("Written_Date"))=S;
                        }
                        else if (CC4(Begin+Info_Offset)==CC4("IDIT"))
                        {
                            Ztring S;
                            S.From_Local((const char*)Begin+Info_Offset+8, Info_XXX_Size);
                            if (S.size()==10)
                                //The specification says to format the date as: "2005-10-14",
                                //however there should be some tolerance allowed.
                                //Note : no conversion from local to UTC, local time of read is not local time of record.
                                General[0](_T("Mastered_Date"))=S.substr(0, 4)+S.substr(5, 2)+S.substr(7, 2);
                            else
                                General[0](_T("Mastered_Date"))=S;
                        }
                        else if (CC4(Begin+Info_Offset)==CC4("IARL"))
                            General[0](_T("Archival_Location")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("IGNR"))
                            General[0](_T("Genre")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("ICMT"))
                            General[0](_T("Comment")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("ISFT"))
                            General[0](_T("Encoded_Application")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("ISRF"))
                            General[0](_T("Encoded_Original")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        else if (CC4(Begin+Info_Offset)==CC4("ICOP"))
                            General[0](_T("Copyright")).From_Local((const char*)Begin+Info_Offset+8, 0, Info_XXX_Size);
                        Info_Offset+=Info_XXX_Size+(Info_XXX_Size&1)+8;
                    }
                }
                //Junk
                else if (CC4(Begin+List_Offset)==CC4("JUNK"))
                {
                    if (CC4(Begin+List_Offset+4)==CC4("DivX"))
                    {
                        General[0](_T("Format"))=_T("DivX");
                        General[0](_T("Format/String"))=_T("DivX");
                        General[0](_T("Format/Extensions"))=_T("DIVX");
                    }
                }
                else if (CC4(Begin+List_Offset)==CC4("movi"))
                {
                }
            }
            else if (CC4(Begin+Avi_Offset)==CC4("idx1"))
            {
            }
            //Junk
            else if (CC4(Begin+Avi_Offset)==CC4("JUNK"))
            {
                if (CC4(Begin+Avi_Offset+8)==CC4("DivX"))
                {
                    General[0](_T("Format"))=_T("DivX");
                    General[0](_T("Format/String"))=_T("DivX");
                    General[0](_T("Format/Extensions"))=_T("DIVX");
                }
            }
            Avi_Offset+=Avi_XXX_Size+(Avi_XXX_Size&1)+8;
        }
        if (!Video.empty())
        {
            if (TotalFrame!=0)
               Video[0](_T("FrameCount")).From_Number(TotalFrame);

            if (Video[0](_T("FrameRate")).To_float32()!=0)
                General[0](_T("PlayTime")).From_Number(((float32)TotalFrame) / Video[0](_T("FrameRate")).To_float32() * 1000, 0);

            //Calculation of missing information
            if (General[0](_T("PlayTime")).To_int32s()!=0)
            {
                General[0](_T("OveralBitRate")).From_Number((int)(8*(float)General[0](_T("FileSize")).To_float32()/(((float)General[0](_T("PlayTime")).To_int32s())/1000)));
                if (General[0](_T("OveralBitRate")).size()>5) //OveralBitRate is > 100 000, to avoid strange behavior
                {
                    Video[0](_T("BitRate")).From_Number(General[0](_T("OveralBitRate")).To_int32s()-5000); //5000 bps because of AVI OverHead (from GordianKnot)
                    for (size_t Pos=0; Pos<Audio.size(); Pos++)
                    {
                        int OverHead=8000; //Classic OverHead (for MP3) (from GordianKnot)
                        if (Audio[Pos](_T("Codec"))==_T("2000")) //AC3
                            OverHead=4750; //OverHead of AC3 in AVI (from GordianKnot)
                        Video[0](_T("BitRate")).From_Number(Video[0](_T("BitRate")).To_int32s()-Audio[Pos](_T("BitRate")).To_int32s()-OverHead);
                    }
                }
            }
        }
    }
    else
        return -1; //WAVE nor AVI
*/
    return 1;
}

//---------------------------------------------------------------------------
void File_Riff::HowTo(stream_t StreamKind)
{
         if (StreamKind==Stream_General)
    {
        General[0](_T("Format"), Info_HowTo)=_T("R");
        General[0](_T("OveralBitRate"), Info_HowTo)=_T("R");
        General[0](_T("PlayTime"), Info_HowTo)=_T("R");
        General[0](_T("Movie"), Info_HowTo)=_T("R INAM");
        General[0](_T("Track"), Info_HowTo)=_T("N INAM");
        General[0](_T("Track/Position"), Info_HowTo)=_T("N IPRT");
        General[0](_T("Album/Track_Total"), Info_HowTo)=_T("N IFRM");
        General[0](_T("Composer"), Info_HowTo)=_T("R IMUS");
        General[0](_T("WrittenBy"), Info_HowTo)=_T("R IWRI");
        General[0](_T("Director"), Info_HowTo)=_T("R IART");
        General[0](_T("DirectorOfPhotography"), Info_HowTo)=_T("R ICNM");
        General[0](_T("EditedBy"), Info_HowTo)=_T("R IEDT");
        General[0](_T("Producer"), Info_HowTo)=_T("R IPRO");
        General[0](_T("ProductionDesigner"), Info_HowTo)=_T("R IPDS");
        General[0](_T("CostumeDesigner"), Info_HowTo)=_T("R ICDS");
        General[0](_T("MasteredBy"), Info_HowTo)=_T("R IENG");
        General[0](_T("ProductionStudio"), Info_HowTo)=_T("R ISTD");
        General[0](_T("DistributedBy"), Info_HowTo)=_T("R IDST");
        General[0](_T("EncodedBy"), Info_HowTo)=_T("R ITCH");
        General[0](_T("CommissionedBy"), Info_HowTo)=_T("R ICMS");
        General[0](_T("Encoded_Original/DistributedBy"), Info_HowTo)=_T("R ISRC");
        General[0](_T("Subject"), Info_HowTo)=_T("R ISBJ");
        General[0](_T("Keywords"), Info_HowTo)=_T("R IKEY");
        General[0](_T("LawRating"), Info_HowTo)=_T("R IRTD");
        General[0](_T("Language"), Info_HowTo)=_T("R ILNG");
        General[0](_T("Medium"), Info_HowTo)=_T("R IMED");
        General[0](_T("Product"), Info_HowTo)=_T("R IPRD");
        General[0](_T("Country"), Info_HowTo)=_T("R ICNT");
        General[0](_T("Written_Date"), Info_HowTo)=_T("R ICRD");
        General[0](_T("Mastered_Date"), Info_HowTo)=_T("R IDIT");
        General[0](_T("Archival_Location"), Info_HowTo)=_T("R IARL");
        General[0](_T("Genre"), Info_HowTo)=_T("R IGNR");
        General[0](_T("Comment"), Info_HowTo)=_T("R ICMT");
        General[0](_T("Encoded_Application"), Info_HowTo)=_T("R ISFT");
        General[0](_T("Encoded_Original"), Info_HowTo)=_T("R ISRF");
        General[0](_T("Copyright"), Info_HowTo)=_T("R ICOP");
    }
    else if (StreamKind==Stream_Video)
    {
        Video[0](_T("Codec"), Info_HowTo)=_T("R");
        Video[0](_T("FrameRate"), Info_HowTo)=_T("R");
        Video[0](_T("FrameCount"), Info_HowTo)=_T("R");
        Video[0](_T("Width"), Info_HowTo)=_T("R");
        Video[0](_T("Height"), Info_HowTo)=_T("R");
        Video[0](_T("AspectRatio"), Info_HowTo)=_T("R");
        Video[0](_T("BitRate"), Info_HowTo)=_T("R");
    }
    else if (StreamKind==Stream_Audio)
    {
        Audio[0](_T("BitRate"), Info_HowTo)=_T("R");
        Audio[0](_T("Channel(s)"), Info_HowTo)=_T("R");
        Audio[0](_T("SamplingRate"), Info_HowTo)=_T("R");
        Audio[0](_T("Codec"), Info_HowTo)=_T("R");
    }
}

} //NameSpace

#endif //MEDIAINFO_RIFF_*








